#!/usr/bin/perl

use IO::File;
use IO::Dir;
use File::stat;
use File::Basename;
use strict;

### Parameters to adapt to local cofiguration
# Name of the Maildir++ directory relative to the path passed as argument
my $mailbox = $ARGV[0]."/";
# Name space used by BINC for private folders, with IMAPdir often = ""
our $namespace = "INBOX.";

### Nothing should need to be changed below here
our $indent = 0;

die("Mailbox doesn't exist")
  if (!-d $mailbox);
parse_mailbox($mailbox);

# Sanity check for namespace
my $subscriptionsSize = -s $mailbox.'/subscriptions';
if ($subscriptionsSize == 0) {
  print $/;
  print $/;
  print "WARNING: Your new subscriptions file is empty. Are you using the
correct namespace? If not re-run script with correct namespace parameter.", $/;
}


sub parse_mailbox
{
  my ($mailbox) = @_;
  print " " x $indent, "Parsing ", $mailbox, " ...", $/;
  $indent += 2;

  my $mb = IO::Dir->new($mailbox)
    or die("Unable to open mailbox $mailbox");
  while(my $file = $mb->read())
  {
    my $absfile = $mailbox."/".$file;
    next
      if ($file eq "." || $file eq "..");
    if ($file eq ".bincimap-subscribed" && -f $absfile)
    {
      convert_subscribtions($absfile);
    }
    elsif ($file eq "bincimap-cache" && -f $absfile)
    {
      convert_cache($absfile);
    }
    elsif (substr($file, 0, 1) eq "." && -d $absfile && -e $absfile."/maildirfolder")
    {
      parse_mailbox($absfile);
    }
  }
  $mb->close();

  $indent -= 2;
  return 1;
}

sub convert_cache
{
  my ($infile) = @_;
  my $dir = dirname($infile);
  my %uids = ();

  print " " x $indent, "Converting cache...", $/;
  my $in = IO::File->new("<".$infile)
     or die("Unable to open cache file $infile");
  my ($blockopen, $uid) = (0, 0);
  my $id = "";
  while(my $line = $in->getline())
  {
    if ($line =~ /^\d+\s{$/)
    {
      $blockopen = 1;
      $uid = 0;
      $id = "";
      next;
    }
    elsif ($blockopen && $line =~ /^}$/)
    {
      $blockopen = 0;
      next;
    }
    elsif ($blockopen && $line =~ /^\t_UID\s=\s(\d+),?$/)
    {
      $uid = $1;
    }
    elsif ($blockopen && $line =~ /^\t_ID\s=\s"?(.*?)"?,?$/)
    {
      $id = $1;
    }
    if ($uid > 0 && length($id) > 0)
    {
      $uids{$uid} = $id;
      $uid = 0;
      $id = "";
      next;
    }
  }
  $in->close();

  if (scalar(keys(%uids)) <= 0)
  {
    print " " x $indent, "Empty uidlist. Skipping...", $/;
    return 1;
  }

  my $uidvalfile = $dir."/bincimap-uidvalidity";
  my ($uidvalidity, $uidnext) = (0, 0);
  die("Error: File $uidvalfile doesn't exist")
    if (!-f $uidvalfile);
  $in = IO::File->new("<".$uidvalfile)
    or die("Unable to open file: $uidvalfile");
  while(my $line = $in->getline())
  {
    if ($line =~ /^\t_uidvalidity\s=\s(\d+),?$/)
    {
      $uidvalidity = $1;
    }
    elsif ($line =~ /^\t_uidnext\s=\s(\d+),?$/)
    {
      $uidnext = $1;
    }
  }
  $in->close();

  die("Error: either uidnext ($uidnext) or uidvalidity ($uidvalidity) is invalid")
    if ($uidnext <= 0 || $uidvalidity <= 0);

  my $version = 1;
  my $outfile = $dir."/dovecot-uidlist";;
  my $out = IO::File->new(">".$outfile)
    or die("Unable to create cache file $outfile");
  $out->print($version, " ", $uidvalidity, " ", $uidnext, $/);
  foreach my $uid (sort{$a <=> $b} (keys(%uids)))
  {
    $out->print($uid, " ", $uids{$uid}, $/);
  }
  $out->close();

  my $stat = stat($infile);
  chown($stat->uid, $stat->gid, $outfile);
  chmod(0600, $outfile);

  return 1;
}

sub convert_subscribtions
{
  my ($infile) = @_;
  my $dir = dirname($infile);
  my @cache = ();

  print " " x $indent, "Converting subscriptions...", $/;
  my $in = IO::File->new("<".$infile)
    or die("Unable to open file: $infile");
  while(my $line = $in->getline())
  {
    next
      if ($line !~ /^$namespace/);
    $line =~ s/^$namespace?//;
    $line =~ s/\n$//;
    $line =~ s/\r$//;
    $line =~ s/\//\./g;
    next
      if (length($line) <= 0);
    next
      if (!-d $dir."/.".$line);
    push(@cache, $line)
      if (scalar(grep{$_ eq $line}(@cache)) <= 0);
  }
  $in->close();

  my $outfile = $dir."/subscriptions";
  my $out = IO::File->new(">".$outfile)
    or die("Unable to create subscriptions file: $outfile");
  foreach my $subscription (@cache)
  {
    $out->print($subscription, $/);
  }
  $out->close();

  my $stat = stat($infile);
  chown($stat->uid, $stat->gid, $outfile);
  chmod(0600, $outfile);

  return 1;
}

