#!/bin/sh
#
# prueba-enlaces.sh
#
# Prueba secuencialmente a dos sitios de Internet a traves de cada interfaz.
# Si ambos estan inalcanzables entonces marca inactiva la interfaz para que
# la ruta en el kernel quede marcada como muerta.
#
# 20110413a

### Variables configurables

# Los sitios de prueba
# Formato: "direccion puerto"
# Ej. "pop.google.com 995"

SITIO1="pop.gmail.com 995"
SITIO2="208.78.96.184 22"

# Las interfaces que vamos a probar. Separadas por espacios.
# Pueden agregarse todas las que se necesiten.
# Ej. INTERFACES="eth1 eth2"
#

INTERFACES="mlink0 mlink1"


NETCAT=/usr/bin/netcat
IFCONFIG=/sbin/ifconfig

DEBUG=1

### Variables no configurables

### Funciones utiles

# log 
# Guarda un mensaje en el log del script
# para enviarse a syslog y por correo al administrador

function log {

logger -t "`basename $0`[$$]" -p daemon.notice $@

}

# valid_ip
# valid_ip IP_ADDRESS
# 
# Prueba si una direccion IP es valida
#

function valid_ip {
case "$*" in
""|*[!0-9.]*) return 1;;
esac

local IFS=.
set -- $*
[ $# -eq 4 ] && [ ${1:-666} -le 255 ] && [ ${2:-666} -le 255 ] && [ ${3:-666} -le 255 ] && [ ${4:-666} -le 255 ]
}



##### Verificaciones de sanidad #####

if [ ! -x $NETCAT ]; then
   echo
   echo "*** Error: $NETCAT no es ejecutable."
   echo "*** Tal vez quieras ajustar la variable NETCAT en este script. "
   echo
   [ $DEBUG -gt 0 ] && log "$NETCAT no es ejecutable"
   exit 1
fi

if [ ! -x $IFCONFIG ]; then
   echo
   echo "*** Error: $IFCONFIG no es ejecutable."
   echo "*** Tal vez quieras ajustar la variable IFCONFIG en este script. "
   echo
   [ $DEBUG -gt 0 ] && log "$IFCONFIG no es ejecutable"
   exit 1
fi


##### Comenzamos las pruebas #####

for interfaz in $INTERFACES; do

   # Obetenemos la direccion de la interfaz y la validamos
   DIRECCION=$(ip addr show $interfaz | grep "inet " | awk '{print $2}' | cut -f1 -d/)
   if ! valid_ip "$DIRECCION"; then
      [ $DEBUG -gt 0 ] && log "La interfaz \"$interfaz\" no parece tener una direccion valida ($DIRECCION)"
      continue
   fi

   log "Activando $interfaz"
   $IFCONFIG $interfaz up

   log "Esperando 15 segundos para que el puerto en $interfaz se active"
   sleep 15

   [ $DEBUG -gt 0 ] && log "Probando $SITIO1, $interfaz (direccion fuente $DIRECCION)"
   if ! $NETCAT -w 15 -s $DIRECCION -z $SITIO1 &> /dev/null; then
      [ $DEBUG -gt 0 ] && log "$SITIO1 ha fallado, $interfaz (direccion fuente $DIRECCION)"
      [ $DEBUG -gt 0 ] && log "Probando $SITIO2, $interfaz (direccion fuente $DIRECCION)"
      if ! $NETCAT -w 15 -s $DIRECCION -z $SITIO2 &> /dev/null; then
         $IFCONFIG $interfaz down
         [ $DEBUG -gt 0 ] && log "$SITIO2 ha fallado, $interfaz (direccion fuente $DIRECCION)"
         [ $DEBUG -gt 0 ] && log "$IFCONFIG $interfaz down"
      else
         [ $DEBUG -gt 0 ] && log "$interfaz ok"
      fi
   else
      [ $DEBUG -gt 0 ] && log "$interfaz ok"
   fi

done

exit 0
