#!/bin/bash
#
# renew-letsencrypt-cert.sh
#
# Obtiene/renueva un certificado firmado por la CA de letsencrypt.org
#
# Copyright 2016 asic-linux, S.A. de C.V.
# Todos los derechos reservados
#




FECHA="20160630"
VERSION="1.0.0"

DEBUG=0
USA_MONDIR=1

######################################################
#                  Funciones utiles                  #
######################################################


MONDIR="/var/log/mon"
MONFILE=${MONDIR}/`basename $0`

termina_error() {
   # Registramos el suceso en MONDIR
   if [ $USA_MONDIR -gt 0 ]; then
      [ -d "$MONDIR" ] || mkdir -p "$MONDIR"
      if [ $? -gt 0 ]; then
         echo "*** Error al crear $MONDIR" >&2
      else
         echo ----- `date` ----- >> $MONFILE
         echo "*** error: $1" >> "$MONFILE"
         if [ -r $ERRFILE ]; then
            cat $ERRFILE >> "$MONFILE"
         fi
      fi
   fi

   # Informamos via stderr
   echo "*** error: $1" >&2
   if [ -r $ERRFILE ]; then
      cat $ERRFILE >&2
   fi

   rm -f $ERRFILE
   exit 1
}


despliega_uso() {
cat << EOF
Uso: $(basename $0) [opciones] <nombre de dominio> <usuario>

Obtiene/renueva un certificado firmado por la CA de letsencrypt.org

Opciones:

   -V             Muestra el numero de version
   -c <config>    Ruta al archivo de configuracion
   -d <ruta>      Directorio de datos de letsencrypt
   -l <URL>       URL del servicio ACME
   -s <servicio1 servicio2 ...> Lista de servicios a reiniciar
   -r <1|0>       Reiniciar servicios
   -t <ruta>      Ruta al ejecutable del cliente acme_tiny


EOF
}


crea_temp() {
   # Archivo temporal
   local retfile=`mktemp ${TMPBASE}/$(basename $0).XXXXXX`
   if [ $? -gt 0 ]; then
      termina_error "Error al crear archivo temporal en $TMPBASE"
      exit 1
   fi

   echo $retfile
}

crea_temp_dir() {
   # Directorio temporal
   local retdir=`mktemp -d ${TMPBASE}/$(basename $0).XXXXXX`
   if [ $? -gt 0 ]; then
      termina_error "Error al crear directorio temporal en $TMPBASE"
      exit 1
   fi

   echo $retdir
}

#######################################################
#                      Defaults                       #
#######################################################

# Direccion del administrador del sistema
[ "$ADMIN" ] || ADMIN="monitor@asic-linux.com.mx"

# Archivo de configuracion
[ "$CONFIGFILE" ] || CONFIGFILE="/etc/$(basename $0).conf"

# Ruta para archivos temporales
[ "$TMPBASE" ] || TMPBASE="/tmp"

# El directorio base de letsencrypt
[ "$LEHOME" ]   || LEHOME="/etc/letsencrypt"

# Ruta al ejecutable de acme_tiny
[ "$ACMETINY" ] || ACMETINY=/usr/bin/acme_tiny.py

# Lista de servicios que deseamos notificar cuando haya
# un nuevo certificado disponible.
# Son nombres de archivos rc, sin el rc.
# Ej. "httpd dovecot qmail"
[ "$RESTART_LIST" ] || RESTART_LIST="httpd"

# Deseamos reiniciar los servicios de RESTART_LIST?
[ "$RESTART_SERVICES" ] || RESTART_SERVICES=1


#######################################################
#              Procesamos las opciones                #
#######################################################


while getopts ":Vc:d:l:s:r:t:h" opt; do
  case $opt in
    V)
      echo "$(basename $0) $VERSION, $FECHA"
      exit 0
      ;;
    c)
      CONFIGFILE="$OPTARG"
      ;;
    d)
      LEHOME="$OPTARG"
      ;;
    l)
      ACMEURL="$OPTARG"
      ATINY_CA_FLAG="--ca $ACMEURL"
      ;;
    s)
      RESTART_LIST="$OPTARG"
      ;;
    r)
      RESTART_SERVICES="$OPTARG"
      ;;
    t)
      ACMETINY="$OPTARG"
      ;;
    h)
      despliega_uso
      exit 1
      ;;
    \?)
      echo "$(basename $0): Opcion invalida: -$OPTARG" >&2
      despliega_uso
      exit 1
      ;;
    :)
      echo "$(basename $0): Opcion -$OPTARG requiere un argumento" >&2
      exit 1
      ;;
  esac
done

shift $(($OPTIND-1))

# Log de errores
ERRFILE=$( crea_temp ) || termina_error "Error al crear el archivo ERRFILE"

# Leemos el archivo de configuracion
if [ -f "$CONFIGFILE" ]; then
   source "$CONFIGFILE" 2>>$ERRFILE || termina_error "Leyendo el archivo de configuracion $CONFIGFILE"
fi

#####################################################
#            Verificaciones de sanidad              #
#####################################################

[ -d "$LEHOME" ]         || termina_error "El directorio de datos de letsencrypt $LEHOME no existe"
[ -d "$LEHOME/certs" ]   || termina_error "El directorio de certificados $LEHOME/certs no existe"
[ -d "$LEHOME/account-keys" ]  || termina_error "El directorio de llaves de cuenta $LEHOME/account-keys no existe"
[ -d "$LEHOME/challenges" ]    || termina_error "El directorio de retos ACME $LEHOME/challenges no existe"
[ -d "$LEHOME/csr" ]     || termina_error "El directorio de solicitudes de firmado $LEHOME/csr no existe"
[ -x "$ACMETINY" ]       || termina_error "$ACMETINY no es ejecutable"




#####################################################
#                 Bucle principal                   #
#####################################################

main () {

NAME=$1
USER=$2


if [ -z "$2" ]; then
    despliega_uso
    exit 1
fi

CERTDIR=${LEHOME}/certs
ACCOUNT_KEY=${LEHOME}/account-keys/${NAME}-account.key
CSR=${LEHOME}/csr/${NAME}.csr
CHALL_DIR=${LEHOME}/challenges/${NAME}

# Solicitamos que firmen nuestro certificado.
# Si la cuenta no existe acme-tiny tratara de crearla automaticamente.
sudo -u ${USER} python  "$ACMETINY" $ATINY_CA_FLAG \
                        --account-key $ACCOUNT_KEY \
                        --csr $CSR \
                        --acme-dir $CHALL_DIR > ${CERTDIR}/${NAME}.signed.cert.tmp

if [ $? -gt 0 ]; then
   termina_error "Error al renovar certificado para $NAME"
fi

# Creamos el certificado encadenado
cat ${CERTDIR}/${NAME}.signed.cert.tmp ${CERTDIR}/letsencrypt-intermediate.pem  > ${CERTDIR}/${NAME}.chained.cert

# Dejamos el nuevo certificado con su nombre definitivo
mv  ${CERTDIR}/${NAME}.signed.cert.tmp ${CERTDIR}/${NAME}.signed.cert

# Reiniciamos los servicios que sea necesario reiniciar
if [ "$RESTART_SERVICES" = 1 ]; then
   for servicio in $RESTART_LIST; do
      /etc/rc.d/rc.${servicio} restart
   done
fi

}

main "$@"
exit $?
