/*
 * Copyright (c) 2007,2008,2009,2010 Mij <mij@bitchx.it>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 * See http://mij.oltrelinux.com/net/dovecot-qmail-vmailmgr/
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <syslog.h>

/*
 * call by dovecot:
 *      me /checkpass/reply/binary
 *
 * call by checkvpw on successful authentication:
 *      me <username> Maildir <checkpass_reply>
 *
 * where:
 * - <me> will be called by checkvpw on successfull auth
 * - <username> serves me, once re-called, for exporting what
 *   user was authenticated
 * - "Maildir" will be replaced by checkvpw with the path of
 *   the user's mailbox
 * - <checkpass_reply> is the dovecot binary we'll run that gathers
 *   our output back into dovecot
 */

/* full path of the "checkvpw" binary */
#define CHECKVPW_BINARY_PATH    "/usr/bin/checkvpw"
/* what maximum length we can expect, as username+password+timestamp */
#define MAXLEN                  600

/* from which file descriptor we read "username\0password\0timestamp\0" (see
 * the checkpassword interface */
#define CHECKPASSWORD_READFD    3

/* checkpassword interface */
#define CHECKPASSWORD_INTERFACE_ERR_BADPASSWORD     1
#define CHECKPASSWORD_INTERFACE_ERR_MISUSED         2
#define CHECKPASSWORD_INTERFACE_ERR_TEMPORARY       111


#define LOGINIT()
#define LOG(x,y)

/*
#define LOGINIT()       openlog("checkvpw-dovecot-wrapper", 0, LOG_AUTH);
#define LOG(x,y)        syslog(LOG_ERR, "%d -- %s\n", x, y);
*/

/* replace all ":" with "::" in string. Return if a replacement occurred. */
int escape_colon(char *string, size_t buflen) {
    int counter, numcolons;


    /* any ':' in original string? */
    for (numcolons = counter = 0; string[counter] != '\0'; counter++) {
        if (string[counter] == ':') numcolons++;
    }
    if (numcolons > 0) {
        char *tmpstr;
        size_t newlen;
        int j; /* index for iteration over second string */

        /* copy string -> tmpstr, duplicating all ':' -> '::' */
        newlen = strlen(string) + numcolons + 1;
        if (newlen > buflen) {
            LOG(1, "buffer to short to accommodate the resulting mailbox path");
            exit(CHECKPASSWORD_INTERFACE_ERR_TEMPORARY);
        }
        tmpstr = (char *)malloc(newlen);
        /* copy everything, but put "::" when ":" is found */
        for (j = counter = 0; string[counter] != '\0'; counter++, j++) {
            tmpstr[j] = string[counter];
            if (string[counter] == ':') {
                /* add another ':' */
                j++;
                tmpstr[j] = ':';
            }
        }
        tmpstr[j] = '\0';

        /* now copy over to original string */
        strcpy(string, tmpstr);
        free(tmpstr);

        return 1;
    }

    return 0;
}


int main(int argc, char *argv[]) {
    char buffer[MAXLEN];
    const char *username;
    const char *password;
    const char *timestamp;
    int auth_data_length;

    int counter;
    int mypipe[2];

    struct stat mailbox_attrs;

    LOGINIT();

    if (getenv("CHECKVPW_WRAPPER_CHILD") != NULL) {
        /* SECOND CALL */
        char *curworkdir;

        LOG(1, "Second instance");
        if (argc < 4) {
            exit(CHECKPASSWORD_INTERFACE_ERR_MISUSED);
        }

        /* export username */
        setenv("USER", argv[1], 1);

        /* export HOME */
        /* relative -> absolute? */
        if (argv[2][0] != '/') {
            /* yes, get absolute */
            curworkdir = getcwd(NULL, 1000);
            if (snprintf(buffer, MAXLEN, "%s/%s", curworkdir, argv[2]) > MAXLEN) {
                LOG(1, "buffer to short to accommodate the resulting mailbox path");
                exit(CHECKPASSWORD_INTERFACE_ERR_TEMPORARY);
            }
            free(curworkdir);
        } else {
            /* no, plain copy over */
            if (strlen(argv[2]) >= sizeof(buffer)) {
                LOG(1, "buffer to short to accommodate the resulting mailbox path");
                exit(CHECKPASSWORD_INTERFACE_ERR_TEMPORARY);
            }
            strcpy(buffer, argv[2]);
        }
        /* escape possible ':' characters before finally exporting */
        escape_colon(buffer, MAXLEN);

        setenv("HOME", buffer, 1);
        LOG(2, "mailbox:");
        LOG(2, buffer);

        /* export uid and gid */
        if (stat(argv[2], & mailbox_attrs) != 0) {
            LOG(CHECKPASSWORD_INTERFACE_ERR_TEMPORARY, "stat() failed");
            exit(CHECKPASSWORD_INTERFACE_ERR_TEMPORARY);
        }

        sprintf(buffer, "%d", mailbox_attrs.st_uid);
        setenv("userdb_uid", buffer, 1);
        LOG(1, "UID:");
        LOG(1, buffer);
        sprintf(buffer, "%d", mailbox_attrs.st_gid);
        setenv("userdb_gid", buffer, 1);
        LOG(1, "GID:");
        LOG(1, buffer);

        execl(argv[3], argv[3], NULL);

        LOG(CHECKPASSWORD_INTERFACE_ERR_MISUSED, "Unable to execute client binary");
        exit(CHECKPASSWORD_INTERFACE_ERR_MISUSED);
    }


    LOG(1, "First instance");
    /* FIRST CALL */
    if (argc < 2) {
        LOG(CHECKPASSWORD_INTERFACE_ERR_MISUSED, "Invalid number of arguments at first call");
        exit(CHECKPASSWORD_INTERFACE_ERR_MISUSED);                /* check arguments */
    }
    if (argv[0][0] != '/') {
        LOG(CHECKPASSWORD_INTERFACE_ERR_MISUSED, "Path of client binary is relative");
        exit(CHECKPASSWORD_INTERFACE_ERR_MISUSED);       /* check we're called with absolute pathname */
    }
    
    /* extract the username, password and timestamp from auth data */
    auth_data_length = 0;
    counter = 0;
    do {
        if (read(CHECKPASSWORD_READFD, & buffer[auth_data_length], 1) != 1) {
            LOG(CHECKPASSWORD_INTERFACE_ERR_MISUSED, "Unexpected error while reading auth data");
            exit(CHECKPASSWORD_INTERFACE_ERR_MISUSED);
        }
        /* count how many complete tokens we got (expect 3) */
        if (buffer[auth_data_length] == '\0') counter++;
        auth_data_length++;
    } while (counter < 3 && auth_data_length < MAXLEN);

    if (counter != 3) {
        LOG(1, "Insufficient buffer length for storing auth data");
        exit(CHECKPASSWORD_INTERFACE_ERR_TEMPORARY);
    }

    /* tokenize */
    username = buffer;
    password = username + strlen(username) + 1;
    timestamp = password + strlen(password) + 1;


    /* export CHECKVPW_WRAPPER_CHILD first, to recognize where we are
     * if we're called again after successful authentication */
    setenv("CHECKVPW_WRAPPER_CHILD", "", 1);

    /*
     * we would now exec() the child, leaving the auth content
     * on CHECKPASSWORD_READFD. But we can't, so we mimic with
     * fork() and wait()
     */
    close(CHECKPASSWORD_READFD);
    pipe(mypipe);
    if (fork() != 0) {
        close(mypipe[1]);
        dup2(mypipe[0], CHECKPASSWORD_READFD);

        /* call
         *
         *  checkvpw <me> <username> Maildir <checkpass_reply>
         *
         * where:
         * - <me> will be called by checkvpw on successfull auth
         * - <username> serves me, once re-called, for exporting what
         *   user was authenticated
         * - "Maildir" will be replaced by checkvpw with the path of
         *   the user's mailbox
         * - <checkpass_reply> is the dovecot binary we'll run that gathers
         *   our output back into dovecot
         */
        LOG(1, "Execute args:");
        LOG(1, CHECKVPW_BINARY_PATH);
        LOG(1, argv[0]);
        LOG(1, username);
        LOG(1, "Maildir");
        LOG(1, argv[1]);
        execl(CHECKVPW_BINARY_PATH, CHECKVPW_BINARY_PATH, argv[0], username, "Maildir", argv[1], NULL);

        /* if we are here, we could not execute checkvpw */
        LOG(CHECKPASSWORD_INTERFACE_ERR_TEMPORARY, "Unable to execute checkvpw");
        exit(CHECKPASSWORD_INTERFACE_ERR_TEMPORARY);
    }

    close(mypipe[0]);
    /* write back authentication data */
    if (write(mypipe[1], buffer, auth_data_length) != auth_data_length) {
        LOG(CHECKPASSWORD_INTERFACE_ERR_TEMPORARY, "Unexpected error while writing auth data to child");
        exit(CHECKPASSWORD_INTERFACE_ERR_TEMPORARY);
    }

    wait(& auth_data_length);
    return WEXITSTATUS(auth_data_length);
}
